import torch

from attacks import Attack
import torch.nn.functional as F

from constants import DEVICE
from utils import cross_entropy_loss, de_normalization, normalization, imshow, imsave


class EMIM(Attack):
    """ EM + MI-FGSM """
    def __init__(self, model, eps=16 / 255, steps=10, decay=1.0, epochs=5):
        """
        :param model: DNN model
        :param eps: the maximum perturbation
        :param steps: the number of iterations
        :param decay: the decay factor
        """
        super().__init__("EMIM", model)
        self.eps = eps
        self.steps = steps
        self.alpha = self.eps / self.steps
        self.decay = decay
        self.epochs = epochs

    def forward(self, images, labels):
        targets = F.one_hot(labels.type(torch.int64), 1000).float().to(DEVICE)
        images_de_normalized = de_normalization(images)
        images_min = torch.clamp(images_de_normalized - self.eps, min=0.0, max=1.0)
        images_max = torch.clamp(images_de_normalized + self.eps, min=0.0, max=1.0)

        g = torch.zeros_like(images)
        for _ in range(self.epochs):
            pre_adv = images.clone()
            for _ in range(self.steps):
                pre_adv_exchanged = torch.stack([x[torch.randperm(3), :, :] for x in pre_adv])

                logits = self.model(pre_adv_exchanged)
                loss = cross_entropy_loss(logits, targets)
                grad = torch.autograd.grad(loss, pre_adv)[0]
                g = self.decay * g + grad / torch.mean(torch.abs(grad), dim=(1, 2, 3), keepdim=True)

                pre_adv_de_normalized = de_normalization(pre_adv)
                pre_adv_de_normalized = torch.clamp(pre_adv_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                                    max=images_max)
                pre_adv = normalization(pre_adv_de_normalized)

        adv = images.clone()
        for _ in range(self.steps):
            logits = self.model(adv)
            loss = cross_entropy_loss(logits, targets)
            grad = torch.autograd.grad(loss, adv)[0]
            g = self.decay * g + grad / torch.mean(torch.abs(grad), dim=(1, 2, 3), keepdim=True)

            adv_de_normalized = de_normalization(adv)
            adv_de_normalized = torch.clamp(adv_de_normalized + self.alpha * torch.sign(g), min=images_min,
                                            max=images_max)
            adv = normalization(adv_de_normalized)

        return adv
